<?php

class TasksController extends Controller
{
	public $layout = '//layouts/column2';
	public $defaultAction = 'index';

	/**
	 * @return array action filters
	 */
	public function filters()
	{
		return array(
			'accessControl',
			array(
				'COutputCache + XajaxGetBuildingData',
				'duration' => 60,
			),
			'postOnly + ajaxBeginTask, ajaxSubmitTask, ajaxSubmitAnswer, ajaxTakeOrder,
				ajaxTakeTask, ajaxAcceptOrderInvitation, assignTask'
		);
	}

	/**
	 * Specifies the access control rules.
	 * This method is used by the 'accessControl' filter.
	 * @return array access control rules
	 */
	public function accessRules()
	{
		return array(
			array('allow', // allow students only
				'actions' => array(
					'ajaxBeginTask', 'ajaxSubmitTask', 'ajaxSubmitAnswer', 'ajaxTakeOrder',
					'ajaxTakeTask', 'ajaxAcceptOrderInvitation', 'ajaxFetchOrders',
					'ajaxFetchTasks', 'ajaxGetCompaniesToTender', 'BLOCK_assignTask', 'fetchBonusTasks',
					'simulateTaskAssignment','ajaxReportSolution', 'ajaxReportAuction'),
				'roles' => array('student')
			),
			array('deny', // deny all users
				'users' => array('*'),
			),
		);
	}

	public function actionAjaxFetchOrders()
	{
		header('Content-type: text/json');
		$result = array();

		try
		{
			if (!Akcja::isAnyInteriorAllowedAndLessonActive('biuroZlecenia1r', 'biuroZlecenia2r'))
				throw new Exception('Akcja zablokowana');

			$kontrahentId = filter_input(INPUT_GET, 'kontrahentId', FILTER_VALIDATE_INT);
			$companyId = Yii::app()->user->id_firmy;

			$lang = filter_input(INPUT_GET, 'lang', FILTER_SANITIZE_STRING);
			$langset = 0;
			if(!empty($lang)) $langset = intval($lang);

			//pobierz kontrahenta
			$kontrahent = Kontrahent::model()->findByPk($kontrahentId);
			if (empty($kontrahent))
				throw new Exception('Kontrahent nie istnieje.');

			//sprawdz czy mozna współpracować z kontrahentem
			if (!$kontrahent->canCooperate($companyId))
				throw new Exception('Firma odmówiła współpracy z powodu słabej reputacji.');

			//pobierz akcje przetargu
			$actionId = Akcja::getLatestTenderActionId(Yii::app()->user);	//moze db expression

			//pobierz przetargi
			$tenders = Yii::app()->db->createCommand('
				select p."id", p."poziomTrudnosci", sum(nvl2(pf."id", 1, 0)) "attended" from "Przetarg" p
				left join "PrzetargGrupa" pg on pg."idPrzetargu" = p."id"
				left join "PrzetargFirma" pf on pf."idGrupy" = pg."id" and pf."idFirmy" = :companyId and pf."status" > 0
				where p."idAkcji" = :idAkcji and p."idKlasy" = :idKlasy and p."idKontrahenta" = :idKontrahenta
        group by p."id", p."poziomTrudnosci"
			')->queryAll(true, array(
				':idAkcji' => $actionId,
				':idKlasy' => Yii::app()->user->id_klasy,
				':idKontrahenta' => $kontrahentId,
				':companyId' => $companyId
			));
			if (empty($tenders))
				throw new Exception('Przetargi są zablokowane');

			//pobierz maksymalny poziom wynikajacy z certyfikatow
			$levels = Firma::getCompanyMaxLevels($companyId);
			$result['maxLevel'] = $levels[$kontrahent->idDziedziny];

			//oznacz przetargi
			$result['data'] = array();
			foreach ($tenders as $tender)
				$result['data'][] = array(
					'id' => $tender['id'],
					'level' => ( ($langset==0) ? Zlecenie::getPoziomString($tender['poziomTrudnosci']) : Zlecenie::getPoziomStringEn($tender['poziomTrudnosci']) ),
					'dziedzina' => $kontrahent->dziedzina->nazwa,
					'attended' => (int)$tender['attended'],
					'enabled' => (int)$tender['poziomTrudnosci'] <= $result['maxLevel'] ? 1 : 0,
				);
		}
		catch (Exception $ex)
		{
			$result['error'] = $ex->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	private function initSeed($companyId, $kontrahentId, $round)
	{
		Yii::app()->db->createCommand("BEGIN
			SYS.DBMS_RANDOM.SEED('$companyId'||'$kontrahentId'||'$round');
		END;")->execute();
	}

	private function resetSeed()
	{
		Yii::app()->db->createCommand('BEGIN
			SYS.DBMS_RANDOM.SEED(SYSDATE);
		END;')->execute();
	}

	public function actionFetchBonusTasks()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		try
		{
			if (!Akcja::isInteriorAllowedAndLessonInactive('zlecenia'))
				throw new Exception('Akcja zablokowana');

			$lang = filter_input(INPUT_GET, 'lang', FILTER_SANITIZE_STRING);

			$round = Yii::app()->user->tura;
			$companyId = Yii::app()->user->id_firmy;

			$bonusTasks = Firma::getBonusTasksListCount($companyId);
			if ($bonusTasks <= 0)
				throw new Exception('Nie możesz pobierać bonusowych zleceń.');

			$actions = Akcja::model()->findAll(array(
				'condition' => '"t"."idLekcji" = :lessonId and "t"."nazwa" = :tag',
				'params' => array(':lessonId' => $round, ':tag' => Akcja::ACTION_MAIL)
			));
			if (empty($actions))
				throw new Exception('Nie możesz pobierac bonusowych zleceń.');

			//maksymalna ilość zlecen w tej turze
			$result['max'] = Firma::getMaximumTasksCount($companyId)
				+ Firma::getBonusTasksCount($companyId);

			//ilosc zlecen pobranych w tej turze
			$result['cnt'] = Firma::getActiveTasksCount($companyId, Yii::app()->user->id_klasy);

			//pobiera reputajce (tj. maksymalne poziomy dla kazdej dziedziny)
			$poziomy = Firma::getReputations($companyId);

			//pobiera "certyfikaty"
			$certs = Firma::getCompanyMaxLevels($companyId);

			$this->initSeed($companyId, -1, $round);

			//każdy może stanąć do przetargu -> pobierz wszyskir w których nie brno udziału
			$params = array(
				':firma' => $companyId, ':max' => $bonusTasks,
				':poziom1' => $poziomy[1], ':poziom2' => $poziomy[2],
				':poziom3' => $poziomy[3], ':poziom4' => $poziomy[4],
			);
			if (!empty($lang)) $params[':lang'] = $lang;
			$zlecenia = Zlecenie::model()->with('dziedzina')->findAllBySql('
				select * from ( select "t".* from "Zlecenie" "t"
					left join "ZlecenieFirmy" "zf" on "zf"."idZlecenia" = "t"."id" and "zf"."idFirmy" = :firma
					where "t"."czyPrzetarg" <= 0  and "zf"."id" is null and "t"."jezyk" '.(empty($lang)?'is null and ':' = :lang and ').' (
						("t"."idDziedziny" = 1 AND "t"."poziom" <= :poziom1) OR
						("t"."idDziedziny" = 2 AND "t"."poziom" <= :poziom2) OR
						("t"."idDziedziny" = 3 AND "t"."poziom" <= :poziom3) OR
						("t"."idDziedziny" = 4 AND "t"."poziom" <= :poziom4)
					) and ("t"."wylaczone" is null or "t"."wylaczone" != 1) ORDER BY dbms_random.value
				) where rownum <= :max', $params);
			if (empty($zlecenia))
				throw new Exception('Brak dostępnych zleceń.');

			$sample = 3;

			$sql = '';
			foreach (range(1, 4) as $idDziedizny) $sql .= '(
				select * from ( select k.* from "Kontrahent" k where k."idDziedziny" = '.$idDziedizny.' and k."typ" = :typ and k."id" not in (
					select kx."idKontrahenta" from "ZlecenieFirmy" kx
					where kx."idFirmy" = :idFirmy and kx."czasZakonczenia" is not null
					group by kx."idKontrahenta"
					having sum(nvl2(kx."idOdpowiedzi", 1, 0)) / count(kx."id") < :threshold
				) order by SYS.DBMS_RANDOM.VALUE) where ROWNUM <= :sample
			)'.($idDziedizny<4?' union all ':'');
			$kontrahents = Kontrahent::model()->findAllBySql($sql, array(
				':typ' => empty($lang) ? Kontrahent::TYPE_STANDARD : Kontrahent::TYPE_FOREIGN,
				':threshold' => (float)ParametryGry::getValue('ZLECENIE_KONTRAHENT_PROG', 0.5),
				':sample' => $sample, ':idFirmy' => $companyId
			));
			if (empty($kontrahents))
				throw new Exception('Brak wolnych kontrahentów.');

			$this->resetSeed();

			$result['data'] = array();
			foreach ($zlecenia as $task)
			{
				$kontrahent = $kontrahents[rand($sample * ($task->idDziedziny-1),
					$sample * $task->idDziedziny - 1)];
				$rec = array(
					'allowed' => $poziomy[$task->idDziedziny] >= $task->poziom,
					'poziom' => $task->poziomStr,
					'dziedzina' => $task->dziedzina->nazwa,
					'idDziedziny' => $task->idDziedziny,
					'cena' => $task->cena,
					'czas' => $task->czas,
					'id' => $task->id,
					'idKontrahenta' => $kontrahent->id,
					'kontrahent' => $kontrahent->attributes,
					'kara' => $task->kara,
					'jezyk' => $task->jezyk
				);
				if ($certs[$task->idDziedziny] >= $task->poziom)
					$rec['kategoria'] = $task->kategoria;
				$result['data'][] = $rec;
			}
			usort($result['data'], function($a, $b){
				return $b['allowed'] - $a['allowed'];
			});

			$workersList = Firma::getWorkersList($companyId);
			$result['workerList'] = CHtml::dropDownList('listAssignWorker', '', $workersList);
			$result['city'] = Mapa::getNameByClass(Yii::app()->user->id_klasy);
			$result['periodDays'] = ParametryGry::getInt('ZLECENIE_ILOSC_DNI', 7);
		}
		catch(Exception $e)
		{
			$this->resetSeed();
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAjaxFetchTasks()
	{
		header('Content-type: text/json');
		$result = array();

		try
		{
			if (!Akcja::isAnyInteriorAllowedAndLessonActive('biuroZlecenia1r', 'biuroZlecenia2r'))
				throw new Exception('Akcja zablokowana');

			$lang = filter_input(INPUT_GET, 'lang', FILTER_SANITIZE_STRING);

			$kontrahentId = filter_input(INPUT_GET, 'kontrahentId', FILTER_VALIDATE_INT);
			$companyId = Yii::app()->user->id_firmy;
			$round = Yii::app()->user->tura;

			$kontrahent = Kontrahent::model()->find('"t"."id" = :kontrahentId', array(
					':kontrahentId' => $kontrahentId
			));
			if (empty($kontrahent))
				throw new Exception('Nie ma podanego kontrahenta.');

			//sprawdz czy mozna współpracować z kontrahentem
			if (!$kontrahent->canCooperate($companyId))
				throw new Exception('Firma odmówiła współpracy z powodu słabej reputacji.');

			//maksymalna ilość zlecen w tej turze
			$result['max'] = Firma::getMaximumTasksCount($companyId);

			//ilosc zlecen pobranych w tej turze
			$result['cnt'] = Firma::getActiveTasksCount($companyId, Yii::app()->user->id_klasy);

			//pobiera reputajce (tj. maksymalne poziomy dla kazdej dziedziny)
			$poziomy = Firma::getReputations($companyId);

			//pobiera ilość zlecen do pobrania
			$maxTasks = Firma::getTasksListCount($companyId);

			//pobiera "certyfikaty"
			$certs = Firma::getCompanyMaxLevels($companyId);

			$this->initSeed($companyId, $kontrahentId, $round);

			// PARAMETERS
			$params = array(
				':firma' => $companyId, ':kontrahentId' => $kontrahentId,
				':poziom1' => $poziomy[1], ':poziom2' => $poziomy[2],
				':poziom3' => $poziomy[3], ':poziom4' => $poziomy[4],
				':max' => $maxTasks
			);
			if (!empty($lang)) $params[':lang'] = $lang;

			//każdy może stanąć do przetargu -> pobierz wszyskir w których nie brno udziału
			$zlecenia = Zlecenie::model()->with('dziedzina')->findAllBySql('
				select * from (select "t".* /*, "dziedzina"."id" "dId", "dziedzina"."nazwa"*/ from "Zlecenie" "t"
				left join "Dziedzina" "dziedzina" on "dziedzina"."id" = "t"."idDziedziny"
				left join "ZlecenieFirmy" "zf" on "zf"."idZlecenia" = "t"."id" and "zf"."idFirmy" = :firma
				where "t"."czyPrzetarg" <= 0 and "t"."jezyk" '.(empty($lang)?'is null and ':'= :lang and ').'
					"zf"."id" is null and "t"."idDziedziny" = (
						select "k"."idDziedziny" from "Kontrahent" "k" where "k"."id" = :kontrahentId
					) and (
						("t"."idDziedziny" = 1 AND "t"."poziom" <= :poziom1) OR
						("t"."idDziedziny" = 2 AND "t"."poziom" <= :poziom2) OR
						("t"."idDziedziny" = 3 AND "t"."poziom" <= :poziom3) OR
						("t"."idDziedziny" = 4 AND "t"."poziom" <= :poziom4)
					)
					and "t"."kategoria" is not null and "t"."czas" is not null and "t"."cena" is not null
				ORDER BY dbms_random.value) where rownum <= :max
			',$params);

			/*
			array(
				':firma' => $companyId, ':kontrahentId' => $kontrahentId,
				':poziom1' => $poziomy[1], ':poziom2' => $poziomy[2],
				':poziom3' => $poziomy[3], ':poziom4' => $poziomy[4],
				':max' => $maxTasks
			));
			*/

			/*
			$englishDomain = array(
				1 => "Mathematics",
				2 => "Physics",
				3 => "Biology",
				4 => "Chemistry",
			);
			//( empty($lang) ? $task->dziedzina->nazwa : $englishDomain[$task->dziedzina->id] ),
			*/

			$this->resetSeed();

			$result['data'] = array();
			foreach ($zlecenia as $task)
			{
				$rec = array(
					'allowed' => $poziomy[$task->idDziedziny] >= $task->poziom,
					'poziom' => ( empty($lang) ? $task->poziomStr : $task->poziomStrEn ),
					'dziedzina' => $task->dziedzina->nazwa,
					'idDziedziny' => $task->idDziedziny,
					'cena' => $task->cena,
					'czas' => $task->czas,
					'id' => $task->id,
					'idKontrahenta' => $kontrahentId,
					'kontrahent' => $kontrahent->attributes,
					'kara' => $task->kara,
					'jezyk' => $task->jezyk
				);
				if ($certs[$task->idDziedziny] >= $task->poziom)
					$rec['kategoria'] = $task->kategoria;
				$result['data'][] = $rec;
			}
			usort($result['data'], function($a, $b){
				return $b['allowed'] - $a['allowed'];
			});

			$workersList = Firma::getWorkersList($companyId);
			$result['workerList'] = CHtml::dropDownList('listAssignWorker', '', $workersList);
			$result['city'] = Mapa::getNameByClass(Yii::app()->user->id_klasy);
			$result['periodDays'] = ParametryGry::getInt('ZLECENIE_ILOSC_DNI', 7);
		}
		catch (Exception $ex)
		{
			$result['error'] = $ex->getMessage();
			$this->resetSeed();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAjaxGetCompaniesToTender()
	{
		header('Content-type: text/json');
		$result = array();

		try
		{
			if (!Akcja::isAnyInteriorAllowedAndLessonActive('biuroZlecenia1r', 'biuroZlecenia2r'))
				throw new Exception('Akcja zablokowana');

			$tenderId = filter_input(INPUT_GET, 'tenderId', FILTER_VALIDATE_INT);
			$companyId = Yii::app()->user->id_firmy;
			$classId = Yii::app()->user->id_klasy;

			//pobiera przetarg
			$tender = Przetarg::model()->with(array('kontrahent'))->findByPk($tenderId);
			if (empty($tender))
				throw new Exception('Przetarg nie istnieje.');

			//pobiera firmy (nie bedace w przetargu i posiadajace certyfikaty)
			$companies = Yii::app()->db->createCommand('
				select "t"."id", "t"."nazwa" from "Firma" "t"
				left join "PrzetargFirma" "pf" on "pf"."idFirmy" = "t"."id" and "pf"."status" > 0 /* nie uczestniczy w prztargu */
				left join "PrzetargGrupa" "pg" on "pg"."id" = "pf"."idGrupy" and "pg"."idPrzetargu" = :tenderId
				where "t"."status" > 0 and "t"."idKlasy" = :idKlasy and "t"."id" <> :currentCompId and (
					select count(*) from "SzkolenieFirmy" sf
					left join "Szkolenie" s on sf."idSzkolenia" = s."id"
					where sf."czySukces" > 0 and sf."dataZakonczenia" is not null and sf."idFirmy" = "t"."id" and
						s."idDziedzina" = :idDziedziny and s."poziom" >= :minPoziom
				) > 0 /* ma wymagane dokumenty */
				group by "t"."id", "t"."nazwa"
				having sum(nvl2("pg"."id", 1, 0)) = 0
			')->queryAll(true, array(
				':tenderId' => $tender->id,
				':idKlasy' => $classId,
				':currentCompId' => $companyId,
				':idDziedziny' => $tender->kontrahent->idDziedziny,
				':minPoziom' => $tender->poziomTrudnosci,
			));

			//usun firmy ze zla reputacja
			foreach ($companies as $id => $company)
				if (!$tender->kontrahent->canCooperate($company['id']))
					unset($companies[$id]);

			$result['companies'] = $companies;
		}
		catch (Exception $ex)
		{
			$result['error'] = $ex->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	protected function &checkTask(ZlecenieFirmy $zadanie)
	{
		if (empty($zadanie))
			throw new Exception('Nie ma takiego zlecenia.');
		if (!$zadanie->canStudentDoTask(Yii::app()->user->id))
			throw new Exception('Nie możesz podjąć się wykonywania tego zadania.');
		return $zadanie;
	}

	/**
	 * @param int $taskId
	 * @return ZlecenieFirmy
	 * @throws Exception
	 */
	protected function &loadTask($taskId)
	{
		$zadanie = ZlecenieFirmy::model()->with('idZlecenia0.odpowiedziZlecenia',
			'idZlecenia0.dziedzina', 'kontrahent')->find(array(
				'join' => '
					left join "PrzetargGrupa" pg on pg."id" = "t"."idGrupyPrzetargowej"
					left join "PrzetargFirma" pf on pf."idGrupy" = pg."id"',
				'condition' => '"t"."id" = :taskId and ("t"."idFirmy" = :companyId or
					(pf."idFirmy" = :companyId and pf."status" >= :status))',
				'params' => array(
					':taskId' => $taskId,
					':companyId' => Yii::app()->user->id_firmy,
					':status' => PrzetargFirma::STATUS_ACCEPTED
				)
		));
		return $this->checkTask($zadanie);
	}

	/**
	 * @param int $taskId
	 * @return ZlecenieFirmy
	 * @throws Exception
	 */
	protected function &loadStartedTask($taskId)
	{
		$zadanie = ZlecenieFirmy::model()->find(array(
			'join' => '
				left join "PrzetargGrupa" pg on pg."id" = "t"."idGrupyPrzetargowej"
				left join "PrzetargFirma" pf on pf."idGrupy" = pg."id"',
			'condition' => '"t"."id" = :taskId and ("t"."idFirmy" = :companyId or
				(pf."idFirmy" = :companyId and pf."status" >= :status)) and
				"t"."czasZakonczenia" IS NULL AND "t"."czasPodjecia" IS NOT NULL',
			'params' => array(
				':taskId' => $taskId,
				':companyId' => Yii::app()->user->id_firmy,
				':status' => PrzetargFirma::STATUS_ACCEPTED
		)));
		return $this->checkTask($zadanie);
	}

	/**
	 * Rozpoczyna / kontynuje wykonywanie zlecenie
	 */
	public function actionAjaxBeginTask()
	{
		header('Content-type: text/json');
		$result = array();

		try
		{
			if (!Akcja::isInteriorAllowedAndLessonInactive('zlecenia'))
				throw new Exception('Akcja zablokowana');

			if (Akcja::currentCompanyHasDebt())
				throw new Exception('Masz zaległe płatnoości. Do czasu ich uregulowania rozwiązywanie zleceń jest zablokowane.');

			$taskId = filter_input(INPUT_POST, 'taskId', FILTER_VALIDATE_INT);
//			$timeStamp = $_POST['timeStamp'];  //filter_input(INPUT_POST, 'timeStamp', FILTER_VALIDATE_INT);
			$timeStamp = time();

			$zadanie = $this->loadTask($taskId);

			if($zadanie->idGrupyPrzetargowej)
				$result['tenderStatus'] = $zadanie->tenderGroup->status;

			$result['status'] = $zadanie->checkStatus();
			if ($result['status'] != ZlecenieFirmy::TASK_NOTSET)
			{
				$result['you'] = $zadanie->idUcznia == Yii::app()->user->id;
				if (!$result['you'] && !empty($zadanie->uczen->imieNazwisko))
					$result['uczen'] = $zadanie->uczen->imieNazwisko;
				if ($result['status'] == ZlecenieFirmy::TASK_FAILED)
					$result['booknote'] = $zadanie->id;
				else if ($result['status'] == ZlecenieFirmy::TASK_SUCCESS)
					$result['invoice'] = $zadanie->idFaktury;
				if (Yii::app()->user->getState('bankrut'))
					$result['bankrut'] = 1;
				throw new Exception('To zlecenie jest już po czasie');
			}

			$result['zadanie'] = array(
				'kategoria' => $zadanie->idZlecenia0->kategoria,
				'tresc' => $zadanie->idZlecenia0->tresc,
				'jednostka' => $zadanie->idZlecenia0->jednostka,
				'czas' => $zadanie->idZlecenia0->czas,
				'poziom' => $zadanie->idZlecenia0->poziomStr,
			);
			if (!empty($zadanie->idZlecenia0->urlObrazka))
				$result['zadanie']['urlObrazka'] = Yii::app()->baseUrl . '/' . $zadanie->idZlecenia0->urlObrazka;
			else
				unset($result['zadanie']['urlObrazka']);
			$result['dziedzina'] = $zadanie->idZlecenia0->dziedzina->nazwa;
			$result['urlDziedzina'] = $zadanie->idZlecenia0->dziedzina->namesAsImageUrl;
			//$result['odpowiedz'] = $zadanie->idZlecenia0->odpowiedziZlecenia[0]->odpowiedz;

			if (empty($zadanie->czasRozpoczecia))
			{
				$zadanie->idUcznia = Yii::app()->user->id;
				$zadanie->czasRozpoczecia = date('d-m-Y H:i:s', $timeStamp); // time());
				$zadanie->save();
				$zadanie->refresh();
			}
			$result['czasRozpoczecia'] = strtotime($zadanie->czasRozpoczecia);
			$result['timeStamp'] = $timeStamp;
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		if (Yii::app()->user->getState('bankrut'))
			$result['bankrut'] = 1;

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAjaxSubmitAnswer()
	{
		header('Content-type: text/json');
		$result = array();

		try
		{
			if (!Akcja::isInteriorAllowedAndLessonInactive('zlecenia'))
				throw new Exception('Akcja zablokowana');

			$taskId = filter_input(INPUT_POST, 'taskId', FILTER_VALIDATE_INT);
			$answer = filter_input(INPUT_POST, 'answer', FILTER_SANITIZE_STRING);

			$zadanie = $this->loadStartedTask($taskId);

			$zadanie->odpowiedz = CHtml::encode($answer);
			$zadanie->idUcznia = Yii::app()->user->id;
			if (!$zadanie->save())
				throw new Exception('Problem podczas zapisywania odpowiedzi.');
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	/**
	 * wysyła odpowiedź na serwer
	 */
	public function actionAjaxSubmitTask()
	{
		header('Content-type: text/json');
		$result = array();

		try
		{
			if (!Akcja::isInteriorAllowedAndLessonInactive('zlecenia'))
				throw new Exception('Akcja zablokowana');

			$taskId = filter_input(INPUT_POST, 'taskId', FILTER_VALIDATE_INT);
			$answer = filter_input(INPUT_POST, 'answer', FILTER_SANITIZE_STRING);
			$userId = Yii::app()->user->id;

			$zadanie = $this->loadTask($taskId);

			if ($zadanie->isOpen && strtotime($zadanie->czasRozpoczecia) + (int)$zadanie->idZlecenia0->czas > time()-15)
			{
				$result['open'] = 1;
				$zadanie->odpowiedz = $answer;
				$zadanie->idUcznia = $userId;
				$zadanie->finish();
			}

			$result['you'] = $zadanie->idUcznia == $userId;
			if (!$result['you'])
				$result['uczen'] = $zadanie->uczen->imieNazwisko;
			$result['status'] = $zadanie->checkStatus();
			if ($result['status'] == ZlecenieFirmy::TASK_FAILED)
				$result['booknote'] = $zadanie->idGrupyPrzetargowej ? null : $zadanie->id;
			else if ($result['status'] == ZlecenieFirmy::TASK_SUCCESS)
				$result['invoice'] = $zadanie->idFaktury ? $zadanie->idFaktury : null;
			if($zadanie->idGrupyPrzetargowej)
				$result['tenderStatus'] = $zadanie->tenderGroup->status;
			if (Yii::app()->user->getState('bankrut'))
				$result['bankrut'] = 1;

			$result['answer'] = $answer;
			$result['przetarg'] = $zadanie->idGrupyPrzetargowej ? 1 : null;

			$arrans = array();
			foreach($zadanie->idZlecenia0->odpowiedziZlecenia as $odpz)
				array_push($arrans, $odpz->odpowiedz);

			$result['answers'] = implode(', ',$arrans);
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		if (Yii::app()->user->getState('bankrut'))
			$result['bankrut'] = 1;

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionSimulateTaskAssignment()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		try
		{
			$taskId = filter_input(INPUT_GET, 'taskId', FILTER_VALIDATE_INT);
			$workerId = filter_input(INPUT_GET, 'workerId', FILTER_VALIDATE_INT);

			$task = Zlecenie::model()->findByPk($taskId);
			if (empty($task))
				throw new Exception('Nie ma takiego zadania.');

			$worker = Pracownik::model()->with('work')->findByPk($workerId);
			if (empty($worker))
				throw new Exception('Nie ma takiego pracownika.');
			if (empty($worker->work))
				throw new Exception('Pracownik nie jest zatrudniony.');

			$result['stats'] = $worker->getTaskEstimation($task);
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAssignTask()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		try
		{
			if (!Akcja::isAnyInteriorAllowed('zlecenia'))
				throw new Exception('Akcja zablokowana.');

			$taskId = filter_input(INPUT_POST, 'taskId', FILTER_VALIDATE_INT);
			$workerId = filter_input(INPUT_POST, 'workerId', FILTER_VALIDATE_INT);
			$studentId = filter_input(INPUT_POST, 'studentId', FILTER_VALIDATE_INT);
			$companyId = Yii::app()->user->id_firmy;

			$task = ZlecenieFirmy::model()->find('"t"."id" = :taskId and
				"t"."idFirmy" = :companyId and "t"."czasRozpoczecia" is null and
				"t"."idGrupyPrzetargowej" is null',
				array(':taskId' => $taskId, ':companyId' => $companyId)
			);
			if (empty($task))
				throw new Exception('Nie ma takiego zadania.');

			$task->idUcznia = null;
			$task->idPracownika = null;
			$task->czasPrzypisania = null;

			$tasksPerWorker = Firma::getTasksPerWorker($companyId);

			if ($workerId)
			{
				$result['stats'] = $task->assignWorker($workerId, $tasksPerWorker);
			}
			else if ($studentId)
			{
				if ($studentId < 0)
					$studentId = Yii::app()->user->id;

				$task->assignStudent($studentId);
			}

			$task->save(false);
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	/**
	 * just take taks (no cancelation)
	 *
	 * @param taksId
	 */
	public function actionAjaxTakeTask()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		try
		{
			if (!Akcja::isAnyInteriorAllowedAndLessonActive('biuroZlecenia1r', 'biuroZlecenia2r'))
			{
				if( !Akcja::isInteriorAllowedAndLessonInactive('zlecenia') )
					throw new Exception('Akcja zablokowana');
			}

			$taskId = filter_input(INPUT_POST, 'taskId', FILTER_VALIDATE_INT);
			$kontrahentId = filter_input(INPUT_POST, 'kontrahentId', FILTER_VALIDATE_INT);
			$studentId = filter_input(INPUT_POST, 'studentId', FILTER_VALIDATE_INT);
			$workerId = filter_input(INPUT_POST, 'workerId', FILTER_VALIDATE_INT);
			$companyId = Yii::app()->user->id_firmy;
			$classId = Yii::app()->user->id_klasy;

			$maxTasks = Firma::getMaximumTasksCount($companyId, $classId);
			if (!Yii::app()->user->tura_aktywna)
				$maxTasks += Firma::getBonusTasksListCount($companyId);

			$activeTasks = Firma::getActiveTasksCount($companyId, $classId);
			if ($activeTasks >= $maxTasks)
				throw new Exception('Nie można pobrać większej ilości zleceń.');

			//sprawdza czy istnieje
			$task = Zlecenie::model()->find(array(
				'join' => 'left join "ZlecenieFirmy" "zf" on  "zf"."idZlecenia" = "t"."id" and "zf"."idFirmy" = :idFirmy',
				'condition' => '"t"."id" = :taskId and "zf"."id" is null and "t"."czyPrzetarg" <= 0 and ("t"."wylaczone" is null or "t"."wylaczone" != 1)',
				'params' => array(':taskId' => $taskId, ':idFirmy' => $companyId)
			));
			if (empty($task))
				throw new Exception('Zlecenie jest niedostępne');

			if ($task->canDoTask($companyId, $kontrahentId))
			{
				$zlecenie = new ZlecenieFirmy;
				$zlecenie->idZlecenia = $task->id;
				$zlecenie->idKontrahenta = $kontrahentId;
				$zlecenie->idFirmy = $companyId;
				$zlecenie->czasPodjecia = new CDbExpression('SYSDATE');

				$tasksPerWorker = Firma::getTasksPerWorker($companyId);

				if ($workerId)
				{
					$zlecenie->assignWorker($workerId, $tasksPerWorker, false);
				}
				else if ($studentId)
				{
					if ($studentId < 0)
						$studentId = Yii::app()->user->id;

					$zlecenie->assignStudent($studentId, $tasksPerWorker);
				}

				$zlecenie->save(false);
			}

			$workersList = Firma::getWorkersList($companyId);
			$result['workerList'] = CHtml::dropDownList('listAssignWorker', '', $workersList);
			$result['city'] = Mapa::getNameByClass(Yii::app()->user->id_klasy);
			$result['periodDays'] = ParametryGry::getInt('ZLECENIE_ILOSC_DNI', 7);
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	/**
	 * applay to order (no cancelation)
	 * invite other companies
	 *
	 * @param taskId
	 * @param intive[]
	 */
	public function actionAjaxTakeOrder()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		$transaction = Yii::app()->db->beginTransaction();
		try
		{
			if (!Akcja::isAnyInteriorAllowedAndLessonActive('biuroZlecenia1r', 'biuroZlecenia2r'))
				throw new Exception('Akcja zablokowana');

			$tenderId = filter_input(INPUT_POST, 'tenderId', FILTER_VALIDATE_INT);
			$value = filter_input(INPUT_POST, 'value', FILTER_VALIDATE_FLOAT);
			$note = filter_input(INPUT_POST, 'note', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
			$invitations = filter_input(INPUT_POST, 'invitations', FILTER_VALIDATE_INT, FILTER_REQUIRE_ARRAY);
			if (!is_array($invitations)) $invitations = array();
			$companyId = Yii::app()->user->id_firmy;
			$classId = Yii::app()->user->id_klasy;

			$companyTenders = PrzetargFirma::model()->findAll(array(
				'join' => '
					left join "PrzetargGrupa" pg on pg."id" = "t"."idGrupy"
					left join "Przetarg" p on p."id" = pg."idPrzetargu"',
				'condition' => '"t"."idFirmy" = :idFirmy and p."id" = :tenderId',
				'params' => array(':tenderId' => $tenderId, ':idFirmy' => $companyId)
			));
			foreach ($companyTenders as $companyTender)
				if ((int)$companyTender->status > PrzetargFirma::STATUS_INVITED)
					throw new Exception('Nie można dwa razy przystąpić do tego samego przetargu.');

			$actionId = Akcja::getLatestTenderActionId(Yii::app()->user);
			if (empty($actionId))
				throw new Exception('Przetargi są niedostępne.');

			$tender = Przetarg::model()->with('kontrahent.dziedzina')->findByAttributes(array(
				'id' => $tenderId,
				'idKlasy' => $classId,
				'idAkcji' => $actionId,
			));
			if (empty($tender))
				throw new Exception('Przetarg nie istnieje.');

			if (!$tender->kontrahent->canCooperate($companyId))
				throw new Exception('Firma odmówiła współpracy z powodu słabej reputacji.');

			$levels = Firma::getCompanyMaxLevels($companyId);
			if ($levels[$tender->kontrahent->idDziedziny] < (int)$tender->poziomTrudnosci)
				throw new Exception('Nie masz wystarczających certyfikatów do podjęcia się przetargu.');

			//odrzuc poprzednie zaproszenia
			foreach ($companyTenders as $companyTender)
			{
				$companyTender->status = PrzetargFirma::STATUS_REFUSED;
				$companyTender->save(false);
			}

			//dodaj przetarg
			$group = new PrzetargGrupa;
			$group->idPrzetargu = $tenderId;
			$group->status = PrzetargGrupa::TENDER_STATUS_ACCEPTATION_REQUEST;
			$group->oferowanaCena = $value;
			$group->notka = $note;
			if (!$group->save())
			{
				$result['errors'] = $group->errors;
				throw new Exception('Nie można dodać przetargu.');
			}

			//dodaj obecna firme do przetargu
			$firma = new PrzetargFirma;
			$firma->idGrupy = $group->id;
			$firma->idFirmy = $companyId;
			$firma->status = PrzetargFirma::STATUS_OWNER;
			if (!$firma->save())
				throw new Exception('Nie można dodać firmy do przetargu.');

			$c = new CDbCriteria();
			$c->compare('"t"."idKlasy"', $classId);
			$c->addInCondition('"t"."id"', $invitations);
			$c->addCondition('"t"."status" > 0');
			$companies = Firma::model()->with('activeUsers')->findAll($c);

			$result['invitations'] = array();
			foreach ($companies as $company)
			{
				//dodaje do przetargu jako zaproszenie
				$invitation = new PrzetargFirma;
				$invitation->idGrupy = $group->id;
				$invitation->idFirmy = $company->id;
				$invitation->status = PrzetargFirma::STATUS_INVITED;
				if (!$invitation->save())
					throw new Exception('Nie można dodać firmy do przetargu.');

				//wysyła zaproszenie
				foreach ($company->activeUsers as $owner)
				{
					$invite = new AkcjaUzytkownika;
					$invite->idAkcji = $actionId;
					$invite->idUzytkownika = $owner->id;
					$invite->dataWyslania = new CDbExpression('SYSDATE');
					$invite->przeczytana = 0;
					$invite->naglowek = 'Zaproszenie do przetargu';
					$template = StringUtils::c('[firma] zaprasza Was do wzięcia udziału w przetargu
						[przetarg] dla [kontrahent].<br/><br/>
						Wartość zlecenia to [cena] zł, do podzału na [firmy].<br/><br/>
						Jeżeli posiadasz odpowiednie kwalifikacje kliknij [linkAkceptuj] aby zaakceptować.<br/>
						Jeżeli nie jesteś [zainteresowana|zainteresowany] kliknij [linkOdrzuc] aby odrzucić.', $owner->plec);
					$przetarg = $tender->kontrahent->dziedzina->nazwa . ' - ' . $tender->poziomStr;
					$link1 = CHtml::link('tutaj', '#', array('accept'=>1,'invitationId'=>$invitation->id));
					$link2 = CHtml::link('tutaj', '#', array('accept'=>0,'invitationId'=>$invitation->id));
					$compCnt = count($companies) + 1;
					$invite->tresc = str_ireplace(array(
						'[firma]', '[przetarg]', '[kontrahent]', '[linkAkceptuj]', '[linkOdrzuc]', '[cena]', '[firmy]'
					), array(
						Yii::app()->user->nazwa_firmy, $przetarg, $tender->kontrahent->nazwa,
						$link1, $link2, number_format($value, 2, ',', ' '), $compCnt . ' ' .
							StringUtils::numberUnitWorld($compCnt, array('firmę', 'firmy', 'firm'))
					), $template);

					$invite->save(false);
				}

				$result['invitations'][] = array('id' => $company->id);
			}

			$transaction->commit();
		}
		catch(Exception $e)
		{
			$transaction->rollback();
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAjaxAcceptOrderInvitation()
	{
		header('Content-type: text/json; charset=utf-8');
		$result = array();

		try
		{
			$invitationId = filter_input(INPUT_POST, 'invitationId', FILTER_VALIDATE_INT);
			$accept = filter_input(INPUT_POST, 'accept', FILTER_VALIDATE_INT);
			$companyId = Yii::app()->user->id_firmy;

			$tender = PrzetargFirma::model()->with('grupa.przetarg.kontrahent.dziedzina', 'grupa.owner.firma.activeUsers')->find(array(
				'condition' => '"t"."id" = :id AND "przetarg"."idKlasy" = :idKlasy AND "t"."idFirmy" = :idFirmy',
				'params' => array(
					':idKlasy' => Yii::app()->user->id_klasy,
					':idFirmy' => $companyId,
					':id' => $invitationId,
				)
			));
			if (empty($tender))
				throw new Exception('Nie ma podanego zaproszenia.');

			if ($tender->status != PrzetargFirma::STATUS_INVITED)
			{
				$messages = array(
					PrzetargFirma::STATUS_ACCEPTED => 'Przetarg został uprzednio zaakceptowany.',
					PrzetargFirma::STATUS_REFUSED => 'Przetarg został uprzednio odrzucony.',
					PrzetargFirma::STATUS_OWNER => 'Nie możesz ' . ($accept ? 'zaakceptować' : 'odrzucić') . ' przetargu, którego jesteś autorem.'
				);
				throw new Exception($messages[$tender->status]);
			}

			$reason = 0;
			$message = 0;

			if ($accept)
			{
				if ($tender->grupa->status > PrzetargGrupa::TENDER_STATUS_ACCEPTATION_REQUEST)
					throw new Exception('Nie możesz zaakceptować zaproszenia ponieważ przetarg został już rozpatrzony przez nauczyciela.');

				if ($tender->grupa->przetarg->kontrahent->canCooperate($companyId))
				{
					$levels = Firma::getCompanyMaxLevels($companyId);
					if ($levels[$tender->grupa->przetarg->kontrahent->idDziedziny] < (int)$tender->grupa->przetarg->poziomTrudnosci)
					{
						$message = 'Twoja firma nie posiada odpowiednich certyfikatów, zaproszenie zostało odrzucone.';
						$reason = 1;
						$accept = 0;
					}
				}
				else
				{
					$message = 'Kontrahent odmówił współpracy z twoją firmą, zaproszenie zostało odrzucone.';
					$reason = 2;
					$accept = 0;
				}
			}

			$tender->status = $accept ? PrzetargFirma::STATUS_ACCEPTED : PrzetargFirma::STATUS_REFUSED;
			$tender->save(false);

			$title = 'Zaproszenie do przetargu ' . ($accept ? 'zaakceptowane' : 'odrzucone');
			$content = 'Firma ' . Yii::app()->user->nazwa_firmy . ' '
				. ($accept ? 'zaakceptowała' : 'odrzuciła') . ' zaproszenie do przetargu '
				. $tender->grupa->przetarg->kontrahent->dziedzina->nazwa . ' - '
				. $tender->grupa->przetarg->poziomStr . ' dla '
				. $tender->grupa->przetarg->kontrahent->nazwa . '.';
			if ($reason === 1)
				$content .= '<br/><br/>Firma nie posiada odpowiednich certyfikatów.';
			else if ($reason === 2)
				$content .= '<br/><br/>Kontrahent odmówił współpracy.';

			foreach ($tender->grupa->owner->firma->activeUsers as $user)
			{
				$invite = new AkcjaUzytkownika;
				$invite->idAkcji = $tender->grupa->przetarg->idAkcji;
				$invite->idUzytkownika = $user->id;
				$invite->dataWyslania = new CDbExpression('SYSDATE');
				$invite->przeczytana = 0;
				$invite->naglowek = $title;
				$invite->tresc = $content;
				$invite->save();
			}

			if (!empty($message))
				$result['error'] = $message;
		}
		catch(Exception $e)
		{
			$result['error'] = $e->getMessage();
		}

		echo json_encode($result);
		Yii::app()->end();
	}

	public function actionAjaxReportSolution()
	{
		$id = $_POST['taskId'];

		$zlecenieFirmy = ZlecenieFirmy::model()->findByPk($id);

		if($zlecenieFirmy === null)
			throw new Exception('Nie ma podanego zlecenia.');

		if($zlecenieFirmy->zgloszone != 1)
		{
                	//wyslanie maila z informacja
                	$subject = "Uczeń zgłosił odwołanie do zadania";
                	$message = 'Jeden z uczniów zgłosił odwołanie. Wejdź do panelu administracyjnego aby zatwierdzić, zresetować lub odrzucić.';
                	$to = 'kontakt@pyramidlab.pl';
                	$headers = 'From: info@kreatywnie.lublin.eu' . "\r\n" .
                        	'Reply-To: info@kreatywnie.lublin.eu' . "\r\n" .
                        	'X-Mailer: PHP/' . phpversion();
                	mail($to, $subject, $message, $headers);
                }

		$zlecenieFirmy->zgloszone = 1;
		$zlecenieFirmy->save(false);

		Yii::app()->end();
	}

	public function actionAjaxReportAuction()
	{
		$taskId = $_POST['taskId'];

		$zlecenie = ZlecenieFirmy::model()->findByPk($taskId);

		if($zlecenie === null)
			throw new Exception('Wystąpił błąd podczas pobierania przetargu.');

		$przetarg = $zlecenie->tenderGroup;

		if($przetarg !== null)
		{
			//wyslanie maila z informacja
			$subject = "Uczeń zgłosił odwołanie do przetargu";
			$message = 'Jeden z uczniów zgłosił odwołanie. Wejdź do panelu administracyjnego aby zatwierdzić, zresetować lub odrzucić.';
			$to = 'kontakt@pyramidlab.pl';
			$headers = 'From: info@kreatywnie.lublin.eu' . "\r\n" .
					'Reply-To: info@kreatywnie.lublin.eu' . "\r\n" .
					'X-Mailer: PHP/' . phpversion();
			mail($to, $subject, $message, $headers);
		}

		$przetarg->zgloszone = 1;
		$przetarg->save(false);

		Yii::app()->end();
	}
}
